from __future__ import annotations

import datetime
import logging
import time

import jwt
import requests
import toml
import tomli
from requests import sessions


class Database:
    def __init__(self, log: logging.Logger):
        self.hw_id: int
        self.jig_id: int | None = None
        self.config: dict
        self.logger = log
        self.token: str

    def __send_request(self, request: requests.Request):
        request.cookies = {"n": self.token}
        prep = request.prepare()
        with sessions.Session() as S:
            return S.send(prep)

    def __get_token(self):
        content = {
            "name": self.config["server"]["name"],
            "passw": self.config["server"]["passw"],
        }
        resp = requests.post(f"{self.config['server']['url']}/auth", json=content)
        self.token = resp.cookies["n"]

    def __is_token_still_valid(self):
        header = jwt.decode(self.token, options={"verify_signature": False})
        margin = datetime.datetime.now(tz=datetime.timezone.utc) + datetime.timedelta(
            seconds=10
        )
        if header["exp"] <= margin.timestamp():
            self.__get_token()

    def __check_if_registered(self):
        try:
            jig_list = self.__send_request(
                requests.Request("GET", f"{self.config['server']['url']}/jigs/")
            )
            self.logger.debug(f"Got jig list from server: \n {jig_list.json()}")
            for x in jig_list.json():
                self.logger.debug(f"item {x['jigid']} name: {x['name']}")
                if x["name"] == self.config["jig"]["name"]:
                    self.jig_id = x["jigid"]
                    self.logger.debug("Found jig!")
                    break
        except Exception as ex:
            self.logger.error(f"Failed: {ex}")

    def __register_jig(self):
        if self.jig_id is None:
            self.logger.debug(
                "Jig not found in server! Registering the jig to the server."
            )
            try:
                ret = self.__send_request(
                        requests.Request(
                        "POST",
                        f"{self.config['server']['url']}/jigs/",
                        json=self.config["jig"],
                        params="",
                 )
                )
                jig = ret.json()
                self.jig_id = jig["jigid"]
            except Exception as ex:
                self.logger.error(f"Failed: to register {ex}")

    def connect(self):
        self.__get_token()
        self.__check_if_registered()
        self.__register_jig()

    def read_config_file(self, path: str):
        self.logger.info(f"Reading configuration file: {path}")
        try:
            with open(path, mode="rb") as fp:
                self.config = tomli.load(fp)
            self.logger.debug(f"Got configuration file: \n {self.config}")
        except toml.TomlDecodeError as ex:
            self.logger.error(f"Configuration file reading failed: {ex}")
        except FileNotFoundError as ex:
            self.logger.error(f"file not found: {ex}")

    def register_device(self, device: Device):
        self.__is_token_still_valid()
        ret = self.__send_request(
            requests.Request(
                "GET",
                f"{self.config['server']['url']}/device/?imei={device.imei}",
            )
        )
        if ret.status_code == 200:
            return ret.json()["deviceid"]
        content = {
            "imei": device.imei,
            "proddate": device.date,
            "hwverid": device.hwid,
            "jigid": self.jig_id,
        }
        self.logger.info(content)
        ret = self.__send_request(
            requests.Request(
                "POST", f"{self.config['server']['url']}/device/", json=content
            )
        )
        if ret.status_code != 200:
            self.logger.error(f"{ret.status_code} {ret.reason} {ret.content}")
            raise Exception
        else:
            resp = ret.json()
            self.logger.info(f"Imei: {resp['imei']} Device: {resp['deviceid']}")
            device.deviceid = resp["deviceid"]
        return device.deviceid

    def print_label(self, device: Device):
        pass


class Device:
    imei: int
    date: str
    hwid: int
    deviceid: int

    def __init__(self, _imei, _hwid):
        self.imei = _imei
        self.date = time.strftime("%Y-%m-%d", time.localtime())
        self.hwid = _hwid

    def add_deviceid(self, _id):
        self.deviceid = _id
