from __future__ import annotations

import base64
import hashlib
import hmac
import logging
import time

import pydantic
import requests
import toml
import tomli
from requests import sessions


class DigestSignature(object):
    def __init__(self, secret, log, header="Sign", algorithm=hashlib.sha3_256):
        self.secret = secret
        self.header = header
        self.algorithm = algorithm
        self.logger = log

    def __call__(self, request: requests.PreparedRequest):
        try:
            if request.body is None:
                sign_string = f"{request.method}\n{request.url}\n\n{request.headers['nonce']}"
            else:
                sign_string = f"{request.method}\n{request.url}\n{request.body.decode()}\n{request.headers['nonce']}"

            self.logger.debug(f"Sign string {sign_string}")
            signature = hmac.new(
                self.secret.encode(), sign_string.encode(), digestmod=self.algorithm
            )
            request.headers[self.header] = signature.hexdigest()
            return request
        except Exception as ex:
            self.logger.debug(f"Signing failed {ex}")
            raise ex
        
# TODO add hmac to the response too
    def check_response(self, resp: requests.Response):
        pass

class Database:
    def __init__(self, log: logging.Logger):
        self.hw_id: int
        self.jig_id: int | None = None
        self.config: dict
        self.logger = log

    def __send_request(self, request: requests.Request):
        request.auth = DigestSignature(self.config["server"]["secret"], self.logger)
        request.headers["nonce"] = str(time.time_ns())
        prep = request.prepare()
        with sessions.Session() as S:
            return S.send(prep)

    def __check_if_registered(self):
        try:
            jig_list = self.__send_request(requests.Request(
                "GET",
                f"http://{self.config['server']['url']}/jigs/"
            ))
            self.logger.debug(f"Got jig list from server: \n {jig_list.json()}")
            for x in jig_list.json():
                self.logger.debug(f"item {x['jigid']} name: {x['name']}")
                if x["name"] == self.config["jig"]["name"]:
                    self.jig_id = x["jigid"]
                    self.logger.debug("Found jig!")
                    break
        except Exception as ex:
            self.logger.error(f"Failed: {ex}")

    def __register_jig(self):
        if self.jig_id is None:
            self.logger.debug(
                "Jig not found in server! Registering the jig to the server."
            )
            ret = self.__send_request(
                requests.Request(
                    "POST",
                    f"http://{self.config['server']['url']}/jigs/",
                    json=self.config["jig"],
                    params="",
                )
            )
            
            self.logger.debug(f"Request returned {ret.headers}")
            self.logger.debug(f"content {ret.content}")


    def connect(self):
        self.__check_if_registered()
        self.__register_jig()

    def read_config_file(self, path: str):
        self.logger.info(f"Reading configuration file: {path}")
        try:
            with open(path, mode="rb") as fp:
                self.config = tomli.load(fp)
            self.logger.debug(f"Got configuration file: \n {self.config}")
        except toml.TomlDecodeError as ex:
            self.logger.error(f"Configuration file reading failed: {ex}")
        except FileNotFoundError as ex:
            self.logger.error(f"file not found: {ex}")

    def register_device(self, device: Device):
        content = {"imei": device.imei, "proddate": device.date, "hwverid": device.hwid, "jigid": self.jig_id}
        self.logger.info(content)
        ret = self.__send_request(requests.Request("POST", f"http://{self.config['server']['url']}/device/", data=content))
        if ret.status_code != 200:
            self.logger.error(f"{ret.status_code} {ret.reason} {ret.content}")
            raise Exception
        else:
            resp = ret.json()
            self.logger.info(f"Imei: {resp['imei']} Device: {resp['deviceid']}")
            device.deviceid = resp["deviceid"]
        return device
    
    def print_label(self, device: Device):
        pass

class Device:
    imei: int
    date: str
    hwid: int
    deviceid: int

    def __init__(self, _imei, _hwid):
        self.imei = _imei
        self.date = time.strftime("%Y-%m-%d", time.localtime())
        self.hwid = _hwid

    def add_deviceid(self, _id):
        self.deviceid = _id
