from sqlalchemy import (
    Column,
    Date,
    ForeignKey,
    Integer,
    LargeBinary,
    String,
    Text,
    TypeDecorator,
    func,
    type_coerce,
)
from sqlalchemy.orm import backref, deferred, relationship

from .database import Base


class Jig(Base):
    __tablename__ = "jigs"

    jigid = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    description = Column(String)
    device = relationship("Device", back_populates="jig")


class Device(Base):
    __tablename__ = "devices"

    deviceid = Column(Integer, primary_key=True, index=True)
    imei = Column(Integer, unique=True, index=True)
    proddate = Column(Date)

    hwverid = Column(Integer, ForeignKey("hwversions.hwverid"))
    jigid = Column(Integer, ForeignKey("jigs.jigid"))
    jig = relationship("Jig", back_populates="device")
    hwversions = relationship("HwVersion", back_populates="device")
    logs = relationship("Log", back_populates="device")


class HwVersion(Base):
    __tablename__ = "hwversions"

    hwverid = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    device = relationship("Device", back_populates="hwversions")


class Log(Base):
    __tablename__ = "logs"

    logid = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    description = Column(Text)
    logdate = Column(Date)

    deviceid = Column(Integer, ForeignKey("devices.deviceid"))
    device = relationship("Device", back_populates="logs")
    picture = relationship("Pic", back_populates="log", cascade="all,delete")


class Pic(Base):
    __tablename__ = "pictures"

    pictureid = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    picturefile = Column(LargeBinary)
    description = Column(Text)

    logid = Column(Integer, ForeignKey("logs.logid"))
    log = relationship("Log", back_populates="picture")


class PasswordType(TypeDecorator):
    impl = String(40)

    def bind_expression(self, bindvalue):
        """Apply a SQL expression to an incoming cleartext value being
        rendered as a bound parameter.

        For this example, this handler is intended only for the
        INSERT and UPDATE statements.  Comparison operations
        within a SELECT are handled below by the Comparator.

        """
        return func.crypt(bindvalue, func.gen_salt("md5"))

    class comparator_factory(String.comparator_factory):
        def __eq__(self, other):
            """Compare the local password column to an incoming cleartext
            password.

            This handler is invoked when a PasswordType column
            is used in conjunction with the == operator in a SQL
            expression, replacing the usage of the "bind_expression()"
            handler.

            """
            # we coerce our own "expression" down to String,
            # so that invoking == doesn't cause an endless loop
            # back into __eq__() here
            local_pw = type_coerce(self.expr, String)
            return local_pw == func.crypt(other, local_pw)


class User(Base):
    __tablename__ = "users"

    userid = Column(Integer, primary_key=True)
    name = Column(String, nullable=False)
    passw = deferred(Column("passw", PasswordType, nullable=False))
