from __future__ import annotations

import base64
import datetime
import hashlib
import logging
import time
from hashlib import sha3_256
from logging.config import dictConfig
from typing import Annotated

import cryptography
import jwt
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import serialization
from fastapi.middleware import Middleware
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from jwt.exceptions import ExpiredSignatureError
from sqlalchemy.orm import Session

from fastapi import (
    Depends,
    FastAPI,
    File,
    Form,
    HTTPException,
    Request,
    Response,
    UploadFile,
)

from . import crud, models, schemas
from .config import LogConfig
from .database import SessionLocal, engine

models.Base.metadata.create_all(bind=engine)

dictConfig(LogConfig().dict())
logger = logging.getLogger("mycoolapp")

app = FastAPI()
app.add_middleware(
    CORSMiddleware,
    allow_origins=[
        "http://192.168.174.220:3000",
        "http://localhost:3000",
        "http://localhost",
        "http://192.168.174.166:3000",
        "http://localho.st",
        "http://localho.st:3000",
        "http://192.168.174.166",
        "http://127.0.0.1:3000",
        "https://localhost",
        "https://localhost:3000",
    ],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
    expose_headers=["*"],
)


def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()


priv_key_f = open(
    "sql_app/id_rsa", "rb"
).read()

pub_key_f = open(
    "sql_app/id_rsa.pub",
    "rb",
).read()

skey = serialization.load_ssh_private_key(
    priv_key_f, password=b"hh", backend=default_backend()
)

pkey = serialization.load_ssh_public_key(pub_key_f, backend=default_backend())


def create_token_for_session(payload):
    return jwt.encode(payload=payload, key=skey, algorithm="RS256")


async def check_token(request: Request):
    try:
        cookie = request.cookies
        logger.info(cookie)
        if cookie:
            resp = jwt.decode(
                jwt=cookie["n"].encode(),
                key=pkey,
                algorithms=["RS256"],
            )
            logger.info(resp)
        else:
            raise HTTPException(401)
    except ExpiredSignatureError as error:
        raise HTTPException(401)


##############################################


@app.get("/device/")
def get_device(
    skip: int = 0,
    limit: int = 10,
    imei: int | None = None,
    deviceid: int | None = None,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    if deviceid:
        device = crud.get_device(db, deviceid)
    elif imei:
        device = crud.get_device_by_imei(db, imei)
    else:
        device = crud.get_devices(db, skip, limit)
    if device is None:
        raise HTTPException(status_code=404)
    return device


@app.get("/logs/")
def get_log(
    skip: int = 0,
    limit: int = 100,
    logid: int | None = None,
    deviceid: int | None = None,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    if logid:
        log = crud.get_log(db, logid)
    elif deviceid:
        log = crud.get_logs(db, deviceid, skip, limit)
    else:
        raise HTTPException(status_code=400)
    if log is None:
        raise HTTPException(status_code=404)
    return log


@app.get("/pic/")
def get_pic(
    logid: int,
    skip: int = 0,
    limit: int = 10,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    pic = crud.get_pictures(db, logid, skip, limit)
    if pic is None:
        raise HTTPException(status_code=404)
    return pic


# võtab seadme logist pildi
@app.get(
    "/pic/file/",
)
def get_image(
    picid: int,
    db: Session = Depends(get_db),
    jwt: int = Depends(check_token),
):
    picture = crud.get_picture(db, picid)
    logger.info(picture)
    if picture is None:
        raise HTTPException(status_code=404)
    return picture
    return Response(content=picture, media_type="image/png")


@app.get("/jigs/")
def get_jig(
    request: Request,
    skip: int = 0,
    limit: int = 10,
    jigid: int | None = None,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    return crud.get_jigs(db, skip, limit)


@app.get("/jig/")
def get_jig_name(
    jigid: int, jwt: int = Depends(check_token), db: Session = Depends(get_db)
):
    return crud.get_jig_name(db, jigid)


@app.get("/hwversions/")
def get_hwversion(
    request: Request,
    skip: int = 0,
    limit: int = 10,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    return crud.get_hwversions(db, skip, limit)


@app.get("/hwversion/")
def get_hw_name(
    hwid: int, jwt: int = Depends(check_token), db: Session = Depends(get_db)
):
    return crud.get_hwversion_name(db, hwid)


############################
# POST REQUESTS


@app.post("/device/")
def add_new_device(
    device: schemas.DeviceCreate,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    db_device = crud.get_device_by_imei(db=db, imei=device.imei)
    if db_device:
        raise HTTPException(status_code=400, detail="Device already in database!")
    new_device = crud.create_device(db, device)
    return new_device


@app.post("/logs/")
def add_new_log(
    log: schemas.LogCreate,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    new_log = crud.create_log(db, log)
    return new_log


@app.post("/pic/file/")
def add_new_picture(
    logid: int,
    file: Annotated[bytes, File()],
    name: str = "",
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    new_pic = crud.create_picture(db, name, file, logid)
    return new_pic


@app.post("/hwversions/")
def add_new_hwversion(
    hw_version: schemas.HwVerCreate,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    new_hwversion = crud.create_hw_version(db, hw_version)
    return new_hwversion


@app.post("/jigs/")
def add_new_jig(
    jig: schemas.JigCreate,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    return crud.create_jig(db, jig)


@app.post("/logs/delete/")
def delete_log(
    log_id: int,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    return crud.remove_log(db, logid=log_id)


@app.post("/pic/delete/")
def delete_picture(
    pic_id: int,
    jwt: int = Depends(check_token),
    db: Session = Depends(get_db),
):
    return crud.remove_picture(db, picid=pic_id)


@app.post("/auth")
async def check_user(
    user: schemas.User,
    db: Session = Depends(get_db),
):
    dbuser = crud.get_user(db, user.name, user.passw)
    if dbuser is None:
        raise HTTPException(status_code=400, detail="Nope!")
    resp = Response()
    resp.set_cookie(
        key="n",
        value=create_token_for_session(
            {
                "name": dbuser.name,
                "iat": datetime.datetime.now(tz=datetime.timezone.utc),
                "exp": datetime.datetime.now(tz=datetime.timezone.utc)
                + datetime.timedelta(minutes=20),
            }
        ),
        httponly=True,
        samesite="strict",
        secure=True,
    )
    logger.info(resp.raw_headers)
    return resp


@app.post("/register")
async def add_user(db: Session = Depends(get_db)):
    db.add(models.User(name="Testing", passw="Testing"))
    db.commit()
