import base64

from sqlalchemy import select
from sqlalchemy.orm import Session

from . import models, schemas

#############################
# get Device


def get_devices(db: Session, skip: int = 0, limit: int = 100):
    rows = db.execute(
        select(models.Device.imei, models.Device.deviceid).offset(skip).limit(limit)
    )
    if rows is None:
        return None
    return [dict(zip(rows.keys(), row)) for row in rows.all()]


def get_device(db: Session, device_id: int):
    return db.scalars(
        select(models.Device).filter(models.Device.deviceid == device_id)
    ).first()


def get_device_by_imei(db: Session, imei: int):
    return db.scalars(select(models.Device).filter(models.Device.imei == imei)).first()


############################
# get Logs


def get_logs(db: Session, device_id: int, skip: int = 0, limit: int = 100):
    rows = db.execute(
        select(models.Log.name, models.Log.logid)
        .filter(models.Log.deviceid == device_id)
        .offset(skip)
        .limit(limit)
    )
    if rows is None:
        return None
    return [dict(zip(rows.keys(), row)) for row in rows.all()]


def get_log(db: Session, logid: int):
    return db.scalars(select(models.Log).filter(models.Log.logid == logid)).first()


###########################
# get pictures


def get_pictures(db: Session, logid: int, skip: int = 0, limit: int = 10):
    rows = db.execute(
        select(models.Pic.pictureid, models.Pic.name)
        .filter(models.Pic.logid == logid)
        .offset(skip)
        .limit(limit)
    )
    if rows is None:
        return None
    return [dict(zip(rows.keys(), row)) for row in rows.all()]


def get_picture(db: Session, pic_id: int):
    base64_pic = db.scalars(
        select(models.Pic.picturefile).filter(models.Pic.pictureid == pic_id)
    ).first()
    return base64_pic


##########################
# get hwversionsYP


def get_hwversions(db: Session, skip: int = 0, limit: int = 10):
    return db.scalars(select(models.HwVersion).offset(skip).limit(limit)).all()


def get_hwversion_name(db: Session, id: int):
    return db.scalars(select(models.HwVersion.name).filter_by(hwverid=id)).first()


##########################
# get jigs


def get_jigs(db: Session, skip: int = 0, limit: int = 10):
    return db.scalars(select(models.Jig).offset(skip).limit(limit)).all()


def get_jig_name(db: Session, id: int):
    return db.scalars(select(models.Jig.name).filter_by(jigid=id)).first()


##########################
# Get user


def get_user(db: Session, username: str, password: str):
    return db.scalars(
        select(models.User).filter_by(name=username).filter_by(passw=password)
    ).first()


##########################
# create device


def create_device(db: Session, device: schemas.DeviceCreate):
    db_device = models.Device(
        imei=device.imei,
        proddate=device.proddate,
        hwverid=device.hwverid,
        jigid=device.jigid,
    )
    db.add(db_device)
    db.commit()
    db.refresh(db_device)
    return db_device


##########################
# create Log


def create_log(db: Session, log: schemas.LogCreate):
    db_log = models.Log(
        name=log.name, description=log.content, logdate=log.date, deviceid=log.deviceid
    )
    db.add(db_log)
    db.commit()
    db.refresh(db_log)
    return db_log


def remove_log(db: Session, logid: int):
    entry = db.scalars(select(models.Log).filter_by(logid=logid)).first()
    if entry == None:
        return 0
    db.delete(entry)
    db.commit()


def remove_picture(db: Session, picid: int):
    entry = db.scalars(select(models.Pic).filter_by(pictureid=picid)).first()
    if entry == None:
        return 0
    db.delete(entry)
    db.commit()


##########################
# create picture


def create_picture(db: Session, name: str, pic: bytes, log_id: int):
    db_pic = models.Pic(
        name=name,
        picturefile=pic,
        logid=log_id,
    )
    db.add(db_pic)
    db.commit()
    db.refresh(db_pic)
    return db_pic


##########################
# create hw version


def create_hw_version(db: Session, hw_ver: schemas.HwVerCreate):
    db_hw_ver = models.HwVersion(name=hw_ver.name)
    db.add(db_hw_ver)
    db.commit()
    db.refresh(db_hw_ver)
    return db_hw_ver


##########################
# create jig


def create_jig(db: Session, jig: schemas.JigCreate):
    db_jig = models.Jig(name=jig.name, description=jig.description)
    db.add(db_jig)
    db.commit()
    db.refresh(db_jig)
    return db_jig
