#include "main.h"

#define I2C_ADDRESS 0x30F

/* SYSCLK = 80 MHz                      */
/* Rise time = 120ns, Fall time = 25ns  */
/* Register values for:                 */
/* fast mode      0x10801541            */
/* standard mode  0x10E0A1E1            */
/* standard mode  0x00D00E28            */
#define I2C_TIMING 0x00D00E28

/* I2C handler declaration */
I2C_HandleTypeDef I2cHandle;

/* Buffer used for receiving */
uint8_t buffer[1];
/* Variables for the communication */
uint32_t num_received_messages = 0;
uint32_t last_received_message = 0;
uint32_t error_count = 0;

void SystemClock_Config(void);
void Error_Handler(void);

/**
 * @brief  Main program
 * @param  None
 * @retval None
 */
int main(void)
{

  HAL_Init();

  /* Configure the system clock to 80 MHz */
  SystemClock_Config();

  /* Configure LED2 */
  BSP_LED_Init(LED2);

  /* Configure UART for logging */
  MX_USART2_UART_Init();

  /*##-1- Configure the I2C peripheral */
  I2cHandle.Instance = I2Cx;
  I2cHandle.Init.Timing = I2C_TIMING;
  I2cHandle.Init.OwnAddress1 = I2C_ADDRESS;
  I2cHandle.Init.AddressingMode = I2C_ADDRESSINGMODE_10BIT;
  I2cHandle.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;
  I2cHandle.Init.OwnAddress2 = 0;
  I2cHandle.Init.OwnAddress2Masks = I2C_OA2_NOMASK;
  I2cHandle.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;
  I2cHandle.Init.NoStretchMode = I2C_NOSTRETCH_DISABLE;

  if (HAL_I2C_Init(&I2cHandle) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
    {
      BSP_LED_On(LED2);
      HAL_Delay(1000);
      BSP_LED_Off(LED2);
    };
  }

  /* Enable the Analog I2C Filter */
  HAL_I2CEx_ConfigAnalogFilter(&I2cHandle, I2C_ANALOGFILTER_ENABLE);

  /* Transmit preamble */
  HAL_UART_Transmit(&huart2, "\n\nRESET\n", 8, 0xFFFF);
  LOG_TRACE("Peripherals initialized");
  LOG_TRACE("Press the button on MASTER");

  /*##-2- Start waiting for test start */
  while (1)
  {
    HAL_I2C_Slave_Receive(&I2cHandle, buffer, 1, HAL_MAX_DELAY) != HAL_OK;
    if (buffer[0] == 0xFE)
    {
      break;
    }
  }
  LOG_TRACE("START");

  /*##-3- Receive messages */
  while (1)
  {
    while (HAL_I2C_Slave_Receive(&I2cHandle, buffer, 1, 100) != HAL_OK)
    {
      error_count++;
      HAL_I2C_DeInit(&I2cHandle);
      HAL_I2C_Init(&I2cHandle);
      break;
    }
    /* Check if message is new */
    if (buffer[0] != last_received_message)
    {
      /* If it is, count up the received messages */
      num_received_messages++;
      last_received_message = buffer[0];
    }
    /* Two back to back 0xFE messages signals test end */
    else if (buffer[0] == 0xFE)
    {
      num_received_messages--;
      LOG_TRACE("%d", num_received_messages);
      break;
    }
  }
  /* Display slave side results */
  LOG_TRACE("ERRORS: %d", error_count);
  LOG_TRACE("DONE");

  while (1)
  {
  }
}

/**
 * @brief  System Clock Configuration
 *         The system Clock is configured as follows :
 *            System Clock source            = PLL (MSI)
 *            SYSCLK(Hz)                     = 80000000
 *            HCLK(Hz)                       = 80000000
 *            AHB Prescaler                  = 1
 *            APB1 Prescaler                 = 1
 *            APB2 Prescaler                 = 1
 *            MSI Frequency(Hz)              = 4000000
 *            PLL_M                          = 1
 *            PLL_N                          = 40
 *            PLL_R                          = 2
 *            PLL_P                          = 7
 *            PLL_Q                          = 4
 *            Flash Latency(WS)              = 4
 * @param  None
 * @retval None
 */
void SystemClock_Config(void)
{
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};

  /* MSI is enabled after System reset, activate PLL with MSI as source */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_MSI;
  RCC_OscInitStruct.MSIState = RCC_MSI_ON;
  RCC_OscInitStruct.MSIClockRange = RCC_MSIRANGE_6;
  RCC_OscInitStruct.MSICalibrationValue = RCC_MSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_MSI;
  RCC_OscInitStruct.PLL.PLLM = 1;
  RCC_OscInitStruct.PLL.PLLN = 40;
  RCC_OscInitStruct.PLL.PLLR = 2;
  RCC_OscInitStruct.PLL.PLLP = 7;
  RCC_OscInitStruct.PLL.PLLQ = 4;
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
      ;
  }

  /* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2
     clocks dividers */
  RCC_ClkInitStruct.ClockType = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
      ;
  }
}

/**
 * @brief  I2C error callbacks.
 * @param  I2cHandle: I2C handle
 * @note   This example shows a simple way to report transfer error, and you can
 *         add your own implementation.
 * @retval None
 */
void HAL_I2C_ErrorCallback(I2C_HandleTypeDef *I2cHandle)
{
  /** Error_Handler() function is called when error occurs.
   * 1- When Slave don't acknowledge it's address, Master restarts communication.
   * 2- When Master don't acknowledge the last data transferred, Slave don't care in this example.
   */
  if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF)
  {
    Error_Handler();
  }
}

/**
 * @brief  This function is executed in case of error occurrence.
 * @param  None
 * @retval None
 */
void Error_Handler(void)
{
}

#ifdef USE_FULL_ASSERT

/**
 * @brief  Reports the name of the source file and the source line number
 *         where the assert_param error has occurred.
 * @param  file: pointer to the source file name
 * @param  line: assert_param error line source number
 * @retval None
 */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif