#include "main.h"

#define I2C_ADDRESS 0x30F

/* SYSCLK = 80 MHz                      */
/* Rise time = 120ns, Fall time = 25ns  */
/* Register values for:                 */
/* fast mode      0x10801541            */
/* standard mode  0x10E0A1E1            */
/* standard mode  0x00D00E28            */
#define I2C_TIMING 0x00D00E28

/* I2C handler declaration */
I2C_HandleTypeDef I2cHandle;

/* Buffer used for transmission */
uint8_t aTxBuffer[1];
/* Variables for the communication */
uint32_t error_count = 0;
uint32_t num_messages = 500000;
uint32_t message_count = 0;
uint32_t ack_error = 0;
uint32_t timeout_error = 0;

void SystemClock_Config(void);
void Error_Handler(void);

/**
 * @brief  Main program
 * @param  None
 * @retval None
 */
int main(void)
{
  HAL_Init();

  /* Configure the system clock to 80 MHz */
  SystemClock_Config();

  /* Configure LED2 */
  BSP_LED_Init(LED2);

  /* Configure UART for logging */
  MX_USART2_UART_Init();

  /*##-1- Configure the I2C peripheral */
  I2cHandle.Instance = I2Cx;
  I2cHandle.Init.Timing = I2C_TIMING;
  I2cHandle.Init.OwnAddress1 = I2C_ADDRESS;
  I2cHandle.Init.AddressingMode = I2C_ADDRESSINGMODE_10BIT;
  I2cHandle.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;
  I2cHandle.Init.OwnAddress2 = 0;
  I2cHandle.Init.OwnAddress2Masks = I2C_OA2_NOMASK;
  I2cHandle.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;
  I2cHandle.Init.NoStretchMode = I2C_NOSTRETCH_DISABLE;

  if (HAL_I2C_Init(&I2cHandle) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
    {
      BSP_LED_On(LED2);
      HAL_Delay(1000);
      BSP_LED_Off(LED2);
    };
  }

  /* Enable the Analog I2C Filter */
  HAL_I2CEx_ConfigAnalogFilter(&I2cHandle, I2C_ANALOGFILTER_ENABLE);

  /* Transmit preamble */
  HAL_UART_Transmit(&huart2, "\n\nRESET\n", 8, 0xFFFF);
  LOG_TRACE("Peripherals initialized");
  LOG_TRACE("Press the button to send");

  /* Configure User push-button */
  BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

  /* Wait for User push-button press */
  while (BSP_PB_GetState(BUTTON_USER) != GPIO_PIN_RESET)
  {
  }

  /* Delay to avoid button bounce */
  HAL_Delay(50);

  /* Wait for User push-button release */
  while (BSP_PB_GetState(BUTTON_USER) != GPIO_PIN_SET)
  {
  }

  /*##-2- Start the transmission process */
  LOG_TRACE("Starting communication - MASTER TRANSMIT");

  /* Hard coded start, notifies the slave */
  aTxBuffer[0] = 0xFE;
  HAL_I2C_Master_Transmit(&I2cHandle, (uint16_t)I2C_ADDRESS, aTxBuffer, 1, HAL_MAX_DELAY);

  /* Send x messages */
  while (message_count < num_messages)
  {
    aTxBuffer[0] = message_count + 1; // Store the message number in the buffer, only the last byte is sent
    message_count++;

    /* Transmit single byte */
    while (HAL_I2C_Master_Transmit(&I2cHandle, (uint16_t)I2C_ADDRESS, aTxBuffer, 1, 100) != HAL_OK)
    {
      /* ACK error */
      if (HAL_I2C_GetError(&I2cHandle) == HAL_I2C_ERROR_AF)
      {
        ack_error++;
        break;
      }
      /* TIMEOUT error */
      else if (HAL_I2C_GetError(&I2cHandle) == HAL_I2C_ERROR_TIMEOUT)
      {
        timeout_error++;
        HAL_I2C_DeInit(&I2cHandle);
        HAL_I2C_Init(&I2cHandle);
        break;
      }
    }
  }
  /* Hard coded end, notifies the slave */
  aTxBuffer[0] = 0xFE;
  HAL_I2C_Master_Transmit(&I2cHandle, (uint16_t)I2C_ADDRESS, aTxBuffer, 1, HAL_MAX_DELAY);
  HAL_I2C_Master_Transmit(&I2cHandle, (uint16_t)I2C_ADDRESS, aTxBuffer, 1, HAL_MAX_DELAY);

  /* Display master side results */
  LOG_TRACE("Transmitted %d messages", num_messages);
  LOG_TRACE("ACK errors: %d", ack_error);
  LOG_TRACE("TIME errors: %d", timeout_error);
  LOG_TRACE("Successful messages: %d", num_messages - ack_error - timeout_error);
}

/**
 * @brief  System Clock Configuration
 *         The system Clock is configured as follows :
 *            System Clock source            = PLL (MSI)
 *            SYSCLK(Hz)                     = 80000000
 *            HCLK(Hz)                       = 80000000
 *            AHB Prescaler                  = 1
 *            APB1 Prescaler                 = 1
 *            APB2 Prescaler                 = 1
 *            MSI Frequency(Hz)              = 4000000
 *            PLL_M                          = 1
 *            PLL_N                          = 40
 *            PLL_R                          = 2
 *            PLL_P                          = 7
 *            PLL_Q                          = 4
 *            Flash Latency(WS)              = 4
 * @param  None
 * @retval None
 */
void SystemClock_Config(void)
{
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};

  /* MSI is enabled after System reset, activate PLL with MSI as source */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_MSI;
  RCC_OscInitStruct.MSIState = RCC_MSI_ON;
  RCC_OscInitStruct.MSIClockRange = RCC_MSIRANGE_6;
  RCC_OscInitStruct.MSICalibrationValue = RCC_MSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_MSI;
  RCC_OscInitStruct.PLL.PLLM = 1;
  RCC_OscInitStruct.PLL.PLLN = 40;
  RCC_OscInitStruct.PLL.PLLR = 2;
  RCC_OscInitStruct.PLL.PLLP = 7;
  RCC_OscInitStruct.PLL.PLLQ = 4;
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
      ;
  }

  /* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2
     clocks dividers */
  RCC_ClkInitStruct.ClockType = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4) != HAL_OK)
  {
    /* Initialization Error */
    while (1)
      ;
  }
}

/**
 * @brief  I2C error callbacks.
 * @param  I2cHandle: I2C handle
 * @retval None
 */
void HAL_I2C_ErrorCallback(I2C_HandleTypeDef *I2cHandle)
{
  if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF)
  {
    Error_Handler();
  }
}

/**
 * @brief  This function is executed in case of error occurrence.
 * @param  None
 * @retval None
 */
void Error_Handler(void)
{
}

#ifdef USE_FULL_ASSERT

/**
 * @brief  Reports the name of the source file and the source line number
 *         where the assert_param error has occurred.
 * @param  file: pointer to the source file name
 * @param  line: assert_param error line source number
 * @retval None
 */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif
